#!/usr/bin/perl
# Perl utility to use Unihedron computer-connected SQM units
# --- Generic modules ---
use Cwd 'abs_path';
# --- GUI modules ---
use Tk;
use Tk::ProgressBar;
use Tk::ROText;
use Tk::NoteBook;
use Tk::Table;
use Tk::Balloon;
use Tk::LabFrame;
use Tk::Trace;
use Tk::StrfClock;
# --- Ethernet modules ---
use IO::Socket;
use IO::Socket::INET;
use IO::Select;
# --- USB modules ---
use if ($^O eq "MSWin32"), Win32::TieRegistry => (qw(:KEY_));
use if ($^O eq "MSWin32"), Win32::SerialPort;
use if ($^O ne "MSWin32"), Device::SerialPort;
use if ($^O eq "MSWin32"), Cwd;

$ConfigOption=0;
$DataLoggingAvailable=0;

foreach (@ARGV) {

	#print help option
	if (substr($_,0,2) eq "-h") {
		print "\n$0 [options]\n";
		print "Description: Sky Quality Meter utility\n";
		print "\t-c\tAllows configuration commands.\n";
		print "\t-d\tAllows debug commands.\n";
		print "\t-L\tExtended logging of all commands and responses.\n";
		print "\t-h\tPrints this help screen.\n";
	}

	#Enable configuration commands
	if (substr($_,0,2) eq "-c") {
		$ConfigOption=1;
	}

	#Enable debug commands
	if (substr($_,0,2) eq "-d") {
		$DebugPageOption=1;
	}

	#Enable Extended logging of all commands and responses
	if (substr($_,0,2) eq "-L") {
		$ExtendedLogging=1;
	}

}

if ($^O eq "MSWin32") {
	$EOL="\r\n";
} else {
	$EOL="\n";
}

$SelectedModel=0;

# Main Window
my $mw = new MainWindow;
$mw->resizable( 0, 0 ); #not resizable in either width or height

#Tooltip/balloon declaration
$b = $mw->Balloon();

#To find usable System Fonts:
#Platform	System fonts
#Unix		(Use xlsfonts for a complete listing)
#Windows	system, systemfixed, ansi, ansifixed, device, oemfixed
#Macintosh	system, application
if ($^O eq "MSWin32"){
	$code_font = $mw->fontCreate(
		'code',
		-family => 'fixedsys',
		-size => 10,
);
} else {
	$code_font = $mw->fontCreate(
		'code',
		-family => 'courier',
		-size => 10,
	);
}


$mw->title("Unihedron SQM Utility");

$mw->optionAdd("*borderWidth", 1); # De-uglification

#GUI Building Area
#--- Devices section -------------------------------------------------
$DeviceFrame=$mw->LabFrame(
	-label=>"Device selection"
)->pack(-anchor=>'w',-fill=>'x');

$DevicesTable=$DeviceFrame->Table(
	-rows=>1,
	-columns=>3,
	-scrollbars=>0,
)->pack(-anchor=>'w');

#--- Find Devices button -------------------------------------------------
$FindDevicesFrame= $DevicesTable -> Frame();
$FindDevicesButton = $FindDevicesFrame -> Button(
	-text => "Find Devices",
	-command =>\&FindDevicesButton_Click,
)->pack;
$DevicesTable->put(0,0,$FindDevicesFrame);
$b->attach($FindDevicesButton,-balloonmsg=>"Find attached USB and Ethernet devices automatically.");

#--- Devices found list -------------------------------------------------
$DevicesFoundFrame= $DevicesTable -> Frame();
$DevicesFound = $DevicesFoundFrame-> Listbox(
	-width=>40,
	-height=>7,
	-font=> $code_font,
	-selectmode=>'single',
);
$DevicesFound->bind('<<ListboxSelect>>',\&DevicesFound_Select);
$DevicesFoundScroll = $DevicesFoundFrame -> Scrollbar(-command => ['yview', $DevicesFound]);
$DevicesFound->configure(-yscrollcommand => ['set', $DevicesFoundScroll]);
$DevicesFound->pack(-in=>$DevicesFoundFrame,-side => 'left', -fill => 'both', -expand => 1);
$DevicesFoundScroll->pack(-in=>$DevicesFoundFrame,-side => 'right', -fill => 'y');
$DevicesTable->put(0,1,$DevicesFoundFrame);

#--- Comm NoteBook -------------------------------------------------
$CommNoteBookFrame=$DevicesTable->Frame();
$CommNoteBook = $CommNoteBookFrame->NoteBook;
$CommNoteBook->pack(-in=>$CommNoteBookFrame,-side=>'left',-fill=>'both',-expand=>1);

$CommPage0 = $CommNoteBook->add("USB",-label=>"USB",-raisecmd=>\&CommPageSel);
$CommPage1 = $CommNoteBook->add("Ethernet",-label=>"Ethernet",-raisecmd=>\&CommPageSel);
$CommPage2 = $CommNoteBook->add("RS232",-label=>"RS232",-raisecmd=>\&CommPageSel);

$DevicesTable->put(0,2,$CommNoteBookFrame);

#--- Status bar -------------------------------------------------
$StatusBar=$mw->Frame();
$ProgressBar = $StatusBar->ProgressBar(
	-width => 18,
	-from => 0,
	-to => 100,
	-gap => 0,
	-variable => \$Progress,
);

$StatusText= $StatusBar->ROText(
	-relief => 'sunken',
	-padx => '4',
	-width =>100,
	-height =>1,
	-font=> $code_font,
);

#--- USB parameters ----------------------------------
$CurrentUSBUnit=$CommPage0->Frame();
$CurrentUSBUnit->pack(-in=>$CommPage0,-side=>'left');
$SerialLabel = $CurrentUSBUnit -> Label(
	-text=>"Serial #:",
);
$SerialNumberText = $CurrentUSBUnit -> ROText (
	-relief => 'sunken',
	-height => 1,
	-width  => 8,
	-font=> $code_font,
);

$PortLabel = $CurrentUSBUnit -> Label(
	-text => "Port:",
);

$PortEntry = $CurrentUSBUnit -> Entry (
	-relief => 'sunken',
	-width => 18,
	-font=> $code_font,
);
$PortEntryBackground = ($PortEntry->configure(-background))[4]; #Get original background color, changed later for error.
$PortEntry->bind('<Button-1>',\&PortEntry_Modify);
$PortEntry->bind('<Key>',\&PortEntry_Modify);

$SerialLabel -> pack(-in=>$CurrentUSBUnit,-side=>'left');
$SerialNumberText ->  pack(-in=>$CurrentUSBUnit,-side=>'left');
$PortLabel ->  pack(-in=>$CurrentUSBUnit,-side=>'left');
$PortEntry ->  pack(-in=>$CurrentUSBUnit,-side=>'left');

#
#--- Ethernet parameters -----------------------------
$CurrentEthUnit=$CommPage1->Frame();
$CurrentEthUnit->pack(-in=>$CommPage1,-side=>'left');
$MACLabel = $CurrentEthUnit -> Label(
	-text=>"MAC:",
);
$MACText = $CurrentEthUnit -> ROText (
	-relief => 'sunken',
	-height => 1,
	-width  => 12,
	-font=> $code_font,
);

$EthIPLabel = $CurrentEthUnit -> Label(
	-text => "IP:",
);
$EthIPEntry = $CurrentEthUnit -> Entry (
	-relief => 'sunken',
	-width => 21,
	-font=> $code_font,
);
$EthIPEntryBackground = ($EthIPEntry->configure(-background))[4]; #Get original background color, changed later for error.
#$EthIPEntry->bind('<<Modified>>',\&EthIPText_Change);
#$EthIPEntry->bind('<Button-1>',\&EthIPText_Change);
$EthIPEntry->bind('<Key>',\&EthIPText_Change);


$EthPortLabel = $CurrentEthUnit -> Label(
	-text => "Port:",
);
$EthPortEntry = $CurrentEthUnit -> Entry (
	-relief => 'sunken',
	-width => 5,
	-font=> $code_font,
);
$EthPortEntryBackground = ($EthPortEntry->configure(-background))[4]; #Get original background color of this text box because it may be changed later for error annunciation.


$MACLabel -> pack(-in=>$CurrentEthUnit,-side=>'left');
$MACText ->  pack(-in=>$CurrentEthUnit,-side=>'left');
$EthIPLabel ->  pack(-in=>$CurrentEthUnit,-side=>'left');
$EthIPEntry ->  pack(-in=>$CurrentEthUnit,-side=>'left');
$EthPortLabel ->  pack(-in=>$CurrentEthUnit,-side=>'left');
$EthPortEntry ->  pack(-in=>$CurrentEthUnit,-side=>'left');

#--- RS232 parameters ----------------------------------
$CurrentRS232Unit=$CommPage2->Frame();
$CurrentRS232Unit->pack(-in=>$CommPage2,-side=>'left');
$RS232PortLabel = $CurrentRS232Unit -> Label(
	-text=>"Port:",
);
$RS232PortEntry = $CurrentRS232Unit -> Entry (
	-relief => 'sunken',
	-width  => 18,
	-font=> $code_font,
);

$RS232BaudLabel = $CurrentRS232Unit -> Label(
	-text => "Baud:",
);

$RS232BaudEntry = $CurrentRS232Unit -> Entry (
	-relief => 'sunken',
	-width => 6,
	-text => 115200,
	-font=> $code_font,
);
$RS232PortEntryBackground = ($RS232PortEntry->configure(-background))[4]; #Get original background color, changed later for error.
#$RS232PortEntry->bind('<Button-1>',\&RS232PortEntry_Modify);
#$RS232PortEntry->bind('<Key>',\&RS232PortEntry_Modify);

#$RS232BaudBackground = ($RS232BaudEntry->configure(-background))[4]; #Get original background color, changed later for error.
#$RS232BaudEntry->bind('<Button-1>',\&RS232BaudEntry_Modify);
#$RS232BaudEntry->bind('<Key>',\&RS232BaudEntry_Modify);

$RS232PortLabel -> pack(-in=>$CurrentRS232Unit,-side=>'left');
$RS232PortEntry ->  pack(-in=>$CurrentRS232Unit,-side=>'left');
$RS232BaudLabel ->  pack(-in=>$CurrentRS232Unit,-side=>'left');
$RS232BaudEntry ->  pack(-in=>$CurrentRS232Unit,-side=>'left');
#=====================================================

$DataNoteBookFrame=$mw->Frame();
$DataNoteBook = $DataNoteBookFrame->NoteBook();

$Page0 = $DataNoteBook->add("page0",-label=>"Information");
$Page1 = $DataNoteBook->add("page1",-label=>"Calibration");
$Page2 = $DataNoteBook->add("page2",-label=>"Report Interval");
$Page3 = $DataNoteBook->add("page3",-label=>"Firmware");
$Page4 = $DataNoteBook->add("page4",-label=>"Data Logging",-raisecmd=>\&LogRefreshButton_Click, -state=>'disabled' );

if($ConfigOption == 1) {
	$Page5 = $DataNoteBook->add("page5",-label=>"Configuration");
}

$Page6 = $DataNoteBook->add("page6",-label=>"GPS", -state=>'disabled' );

if($DebugPageOption == 1) {
	$Page7 = $DataNoteBook->add("page7",-label=>"Debug");
}
#=== Information page ========================================================
$Page0Frame=$Page0->Frame();
$ReadingFrame=$Page0Frame->Frame();
$ReadingButton = $ReadingFrame-> Button(
	-text => "Reading",
	-command =>\&ReadingButton_Click,
);
$ReadingResult = $ReadingFrame-> ROText (
	-relief => 'sunken',
	-height =>6,
	-width => 30,
	-font=> $code_font,
);

$VersionFrame=$Page0Frame->Frame();
$VersionButton = $VersionFrame-> Button(
	-text => "Version",
	-command =>\&VersionButton_Click,
);
$VersionResult = $VersionFrame-> ROText (
	-relief => 'sunken',
	-height =>6,
	-width => 25,
	-font=> $code_font,
);

# === Calibration page =======================================================
$CIB=$Page1->Button(-text=> "Get Calibration Info",-command =>\&CalibrationInfoButton_Click);
$CLB=$Page1->Button(-text=> "Log Calibration Info",-command =>\&CalibrationLogButton_Click);

# Calibration page : Table
$CalibrationTable=$Page1->Table(
	-rows=>5,
	-columns=>4,
	-scrollbars=>0,
);
# Calibration page : Labels
$CalibrationTable->put(1,0,$CalibrationTable->Label(-text=>"Light Calibration Offset",-anchor=>'e'));
$CalibrationTable->put(2,0,$CalibrationTable->Label(-text=>"Light Calibration Temperature",-anchor=>'e'));
$CalibrationTable->put(3,0,$CalibrationTable->Label(-text=>"Dark Calibration Period",-anchor=>'e'));
$CalibrationTable->put(4,0,$CalibrationTable->Label(-text=>"Dark Calibration Temperature",-anchor=>'e'));
 # Calibration page : Entry Fields
$CalibrationTable->put(0,1,$CalibrationTable->Label(-text=>"Desired values"));
$SCLOTF=$CalibrationTable->Entry(-relief => 'sunken',-width => 12,-font=> $code_font,);
$SLCTTF=$CalibrationTable->Entry(-relief => 'sunken',-width => 12,-font=> $code_font,);
$SDCPTF=$CalibrationTable->Entry(-relief => 'sunken',-width => 12,-font=> $code_font,);
$SDCTTF=$CalibrationTable->Entry(-relief => 'sunken',-width => 12,-font=> $code_font,);
$CalibrationTable->put(1,1,$SCLOTF);
$CalibrationTable->put(2,1,$SLCTTF);
$CalibrationTable->put(3,1,$SDCPTF);
$CalibrationTable->put(4,1,$SDCTTF);
# # Calibration page : Buttons
$CalibrationTable->put(1,2,$CalibrationTable->Button(-text => "Set",-command =>\&SetLightCalOffsetButton_Click));
$CalibrationTable->put(2,2,$CalibrationTable->Button(-text => "Set",-command =>\&SetLightCalTempButton_Click));
$CalibrationTable->put(3,2,$CalibrationTable->Button(-text => "Set",-command =>\&SetDarkCalPerButton_Click));
$CalibrationTable->put(4,2,$CalibrationTable->Button(-text => "Set",-command =>\&SetDarkCalTempButton_Click));
# # Calibration page : Results
$CalibrationTable->put(0,3,$CalibrationTable->Label(-text=>"Actual values"));
$SCLOR=$CalibrationTable->ROText(-relief => 'sunken',-height =>1,-width => 12,-font=> $code_font,);
$SLCTR=$CalibrationTable->ROText(-relief => 'sunken',-height =>1,-width => 12,-font=> $code_font,);
$SDCPR=$CalibrationTable->ROText(-relief => 'sunken',-height =>1,-width => 12,-font=> $code_font,);
$SDCTR=$CalibrationTable->ROText(-relief => 'sunken',-height =>1,-width => 12,-font=> $code_font,);
$CalibrationTable->put(1,3,$SCLOR);
$CalibrationTable->put(2,3,$SLCTR);
$CalibrationTable->put(3,3,$SDCPR);
$CalibrationTable->put(4,3,$SDCTR);

$CalibrationNote=$Page1->Label(-text=>"  Note: See calibration sheet for original settings.",-anchor=>'w');

$CIB->pack(-in=>$Page1,-anchor=>'w');
$CLB->pack(-in=>$Page1,-anchor=>'w');
$CalibrationTable->pack(-in=>$Page1,-anchor=>'w');
$CalibrationNote->pack(-in=>$Page1,-anchor=>'w');


# === Report Interval page ==========================================================
$IIB=$Page2->Button(-text=> "Get Report Interval Info",-command =>\&IntervalInfoButton_Click);
# Interval page : Table
$IntervalTable=$Page2->Table(
	-rows=>3,
	-columns=>6,
	-scrollbars=>0,
);
# Report Interval page : Labels
$IntervalTable->put(1,0,$IntervalTable->Label(-text=>"Report Interval Time",-anchor=>'e'));
$IntervalTable->put(2,0,$IntervalTable->Label(-text=>"Report Threshold",-anchor=>'e'));
# Interval page : Entry Fields
$IntervalTable->put(0,1,$IntervalTable->Label(-text=>"Desired values"));
$ITiTF=$IntervalTable->Entry(-relief => 'sunken',-width => 12,-font=> $code_font,);
	$IntervalTable->put(1,1,$ITiTF);
	$b->attach($ITiTF,-balloonmsg=>"Enter reporting interval time in seconds.");
$IThTF=$IntervalTable->Entry(-relief => 'sunken',-width => 12,-font=> $code_font,);
	$IntervalTable->put(2,1,$IThTF);
	$b->attach($IThTF,-balloonmsg=>"Enter reporting threshold in mpsas.");
# Report Interval page : Buttons
$IntervalTimeERButton=$IntervalTable->Button(-text => "ER",-command =>\&IntervalTimeERButton_Click);
	$IntervalTable->put(1,2,$IntervalTimeERButton);
	$b->attach($IntervalTimeERButton,-balloonmsg=>"Copy Interval time to EEPROM and RAM");
$IntervalTimeRButton=$IntervalTable->Button(-text => "R",-command =>\&IntervalTimeRButton_Click);
	$IntervalTable->put(1,3,$IntervalTimeRButton);
	$b->attach($IntervalTimeRButton,-balloonmsg=>"Copy Interval time to RAM");
$IntervalThresholdERButton=$IntervalTable->Button(-text => "ER",-command =>\&ThresholdERButton_Click);
	$IntervalTable->put(2,2,$IntervalThresholdERButton);
	$b->attach($IntervalThresholdERButton,-balloonmsg=>"Copy Threshold to EEPROM and RAM");
$IntervalThresholdRButton=$IntervalTable->Button(-text => "R",-command =>\&ThresholdRButton_Click);
	$IntervalTable->put(2,3,$IntervalThresholdRButton);
	$b->attach($IntervalThresholdRButton,-balloonmsg=>"Copy Threshold to RAM");
# Report Interval page : Results
$IntervalTable->put(0,4,$IntervalTable->Label(-text=>"EEPROM"));
$IntervalTable->put(0,5,$IntervalTable->Label(-text=>"RAM"));
$ITiE=$IntervalTable->ROText(-relief => 'sunken',-height =>1,-width => 12,-font=> $code_font,);
$ITiR=$IntervalTable->ROText(-relief => 'sunken',-height =>1,-width => 12,-font=> $code_font,);
$IThE=$IntervalTable->ROText(-relief => 'sunken',-height =>1,-width => 12,-font=> $code_font,);
$IThR=$IntervalTable->ROText(-relief => 'sunken',-height =>1,-width => 12,-font=> $code_font,);
$IntervalTable->put(1,4,$ITiE);
$IntervalTable->put(1,5,$ITiR);
$IntervalTable->put(2,4,$IThE);
$IntervalTable->put(2,5,$IThR);

# === Firmware page ==========================================================
$FirmwareTable=$Page3->Table(
	-rows=>3,
	-columns=>2,
	-scrollbars=>0,
);
$FirmwareTable->put(0,0,$FirmwareTable->Button(-text => "Select Firmware",-command =>\&SelectFirmwareButton_Click));
$SFR=$FirmwareTable->Text(-relief => 'sunken',-height =>1,-width => 60,-font=> $code_font,);
$SFRBackground = ($SFR->configure(-background))[4];
$FirmwareTable->put(0,1,$SFR);

$FirmwareTable->put(1,0,$FirmwareTable->Button(-text => "Check Lock",-command =>\&CheckLockButton_Click));
$CheckLockResult=$FirmwareTable->ROText(-relief => 'sunken',-height =>1,-width => 60,-font=> $code_font,);
$FirmwareTable->put(1,1,$CheckLockResult);

$FirmwareTable->put(2,0,$FirmwareTable->Button(-text => "Load Firmware",-command =>\&LoadFirmwareButton_Click));

# === Data Logging page ======================================================
# --- Split the page into three columns --------------------------------------
$DataLoggingPageTable=$Page4->Table(
	-rows=>1,
	-columns=>3,
	-scrollbars=>0,
);
$DataLoggingPageTable->pack(-in=>$Page4,-anchor=>'w');

# --- Put frames into the columns for packing other items --------------------
$DataLoggingFrame1=$DataLoggingPageTable->Frame();
	$DataLoggingPageTable->put(0,0,$DataLoggingFrame1);
$DataLoggingFrame2=$DataLoggingPageTable->Frame();
	$DataLoggingPageTable->put(0,1,$DataLoggingFrame2);
$DataLoggingFrame3=$DataLoggingPageTable->Frame();
	$DataLoggingPageTable->put(0,2,$DataLoggingFrame3);

# --- Control frame includes control buttons ---------------------------------
$DataLoggingControlFrame=$DataLoggingFrame3->LabFrame(
	-label=>"Control"
)->pack;
# Refresh button
$LIIB=$DataLoggingControlFrame->Button(-text=> "Refresh",-command =>\&LogRefreshButton_Click)->pack;
	$b->attach($LIIB,-balloonmsg=>"Get data logging settings from the SQM.");

# Apply button
$LogSettingsApplyButton=$DataLoggingControlFrame->Button(-text => "Apply",-command =>\&LogSettingsApplyButton_Click)->pack(-fill=>'x');
	$b->attach($LogSettingsApplyButton,-balloonmsg=>"Apply the data logging settings to the SQM.");

# --- Real time clock display/control ----------------------------------------
$LoggingTimeFrame=$DataLoggingFrame1->LabFrame(
	-label=>"Device Clock"
)->pack(-fill=>'x');

$LoggingTimeTable=$LoggingTimeFrame->Table(
	-rows=>1,
	-columns=>2,
	-scrollbars=>0,
)->pack(-anchor=>'w');

$LRTCSET=$LoggingTimeTable->Button(
	-text=> "Set",
	-command =>\&RTCSetButton_Click
);

$LoggingTimeTable->put(0,0,$LRTCSET);

$b->attach(
	$LRTCSET,
	-balloonmsg=>"Copy time from this PC to the SQM Real Time Clock."
);

$LRTCDateTimeFrame=$LoggingTimeTable->Frame();

$LoggingTimeTable->put(0,1,$LRTCDateTimeFrame);

$LRTCTime=$LRTCDateTimeFrame->ROText(
	-relief => 'sunken',
	-height =>1,
	-width => 19,
	-font=> $code_font
)->pack;

$b->attach(
	$LRTCTime,
	-balloonmsg=>"Snapshot of time from the SQM Real Time Clock."
);

$LRTPCTime=$LRTCDateTimeFrame->StrfClock(
	-font=> $code_font,
	-format=>'%y-%m-%d %w %H:%M:%S')->pack;

# --- Trigger settings -------------------------------------------------------
$LoggingLogTimeSettingsFrame=$DataLoggingFrame1->LabFrame(
	-label=>"Trigger"
)->pack;

$LogIntervalTimesTable=$LoggingLogTimeSettingsFrame->Table(
	-rows=>3,
	-columns=>3,
	-scrollbars=>0,
)->pack;
# Radio button for "Off" selection
$LIROff=$LogIntervalTimesTable->Radiobutton(-text=>"Off",-anchor=>'w',-variable=>\$LogMode,-value=>0);
	$LogIntervalTimesTable->put(0,0,$LIROff);

# Radio button for "Every x seconds" selection
$LIRSeconds=$LogIntervalTimesTable->Radiobutton(-text=>"Every",-anchor=>'w',-variable=>\$LogMode,-value=>1);
	$LogIntervalTimesTable->put(1,0,$LIRSeconds);
$LISecondsE=$LogIntervalTimesTable->Entry(-relief => 'sunken',-width => 5,-font=> $code_font,-justify=>'right');
	$LogIntervalTimesTable->put(1,1,$LISecondsE);
	$b->attach($LISecondsE,-balloonmsg=>"Enter logging interval time in seconds.");
$LISecLabel	= $LogIntervalTimesTable->Label(-text=>"seconds  (always on)",-anchor=>'w');
	$LogIntervalTimesTable->put(1,2,$LISecLabel);

# Radio button for "Every x minutes" selection
$LIRMinutes=$LogIntervalTimesTable->Radiobutton(-text=>"Every",-anchor=>'w',-variable=>\$LogMode,-value=>2);
	$LogIntervalTimesTable->put(2,0,$LIRMinutes);
$LIMinutesE=$LogIntervalTimesTable->Entry(-relief => 'sunken',-width => 5,-font=> $code_font,-justify=>'right');
	$LogIntervalTimesTable->put(2,1,$LIMinutesE);
	$b->attach($LIMinutesE,-balloonmsg=>"Enter logging interval time in minutes.");
$LIMinLabel	= $LogIntervalTimesTable->Label(-text=>"minutes  (power save mode)",-anchor=>'w');
	$LogIntervalTimesTable->put(2,2,$LIMinLabel);

$LogIntervalTimesTable2=$LoggingLogTimeSettingsFrame->Table(
	-rows=>5,
	-columns=>1,
	-scrollbars=>0,
)->pack(-fill=>'x');

# Radio button for "Every 5 minutes" selection
$LIR5Minutes=$LogIntervalTimesTable2->Radiobutton(-text=>"Every 5 minutes on the 1/12th hour",-anchor=>'w',-variable=>\$LogMode,-value=>3);
	$LogIntervalTimesTable2->put(0,0,$LIR5Minutes);
$LIR10Minutes=$LogIntervalTimesTable2->Radiobutton(-text=>"Every 10 minutes on the 1/6th hour",-anchor=>'w',-variable=>\$LogMode,-value=>4);
	$LogIntervalTimesTable2->put(1,0,$LIR10Minutes);
$LIR15Minutes=$LogIntervalTimesTable2->Radiobutton(-text=>"Every 15 minutes on the 1/4 hour",-anchor=>'w',-variable=>\$LogMode,-value=>5);
	$LogIntervalTimesTable2->put(2,0,$LIR15Minutes);
$LIR30Minutes=$LogIntervalTimesTable2->Radiobutton(-text=>"Every 30 minutes on the 1/2 hour",-anchor=>'w',-variable=>\$LogMode,-value=>6);
	$LogIntervalTimesTable2->put(3,0,$LIR30Minutes);
$LIR60Minutes=$LogIntervalTimesTable2->Radiobutton(-text=>"Every hour on the hour",-anchor=>'w',-variable=>\$LogMode,-value=>7);
	$LogIntervalTimesTable2->put(4,0,$LIR60Minutes);

$LogIntervalTimesTable->traceVariable(\$LogMode,'w'=>\&LogModeChanged);
	
sub LogModeChanged {
	my ($index,$value,$op,@args) = @_;
	if (($value eq 0) or ($value ge 2)) { #Off
		$LISecondsE->configure(-state=>'disabled');
		$LIMinutesE->configure(-state=>'disabled');
	}
	if ($value eq 1) { # Seconds
		$LISecondsE->configure(-state=>'normal');
		$LIMinutesE->configure(-state=>'disabled');
	}
	if ($value eq 2) { # Minutes
		$LISecondsE->configure(-state=>'disabled');
		$LIMinutesE->configure(-state=>'normal');
	}
	if ($DataLoggingAvailable eq 1) {
		LogSettingsApplyButton_Click();
	}
	return $value;
}
# --- Threshold settings -----------------------------------------------------
$LoggingThresholdSettingsFrame=$DataLoggingFrame1->LabFrame(
	-label=>"Threshold"
)->pack(-fill=>'x');
$LogThresholdTable=$LoggingThresholdSettingsFrame->Table(
	-rows=>1,
	-columns=>2,
	-scrollbars=>0,
)->pack(-anchor=>'w');
$LIThresholdE=$LogThresholdTable->Entry(-relief => 'sunken',-width => 6,-font=> $code_font,-justify=>'right');
	$LogThresholdTable->put(0,0,$LIThresholdE);
	$b->attach($LIThresholdE,-balloonmsg=>"Enter logging threshold in magnitudes per square arc second.");
$LIThL=$LogThresholdTable->Label(-text=>'mpsas');
	$LogThresholdTable->put(0,1,$LIThL);

	

# --- Storage ----------------------------------------------------------------
$LogCurrentRecord=0;
$LogDBMax=2**20/16;

$LoggingStorageFrame=$DataLoggingFrame2->LabFrame(
	-label=>"Storage"
)->pack(-fill=>'x');

$LogStorageControlFrame=$LoggingStorageFrame->Frame()->pack;

# Retrieve all records button
$LogRetrieveAllButton=$LogStorageControlFrame->Button(
	-text=> "Retrieve all",
	-command => sub {

			# Check if there are any records to retrieve
			
			#Allow retrieval
			$CancelLogRetrieval=0;

			# Get timestamp for filename
			($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=localtime(time);
			$LogDBRecFile=sprintf("%4d%02d%02d-%02d%02d%02d.txt",$year+1900,$mon+1,$mday,$hour,$min,$sec);
			
			$LogRecordResult->Contents("Database records being saved to:\n$LogDBRecFile\nPlease wait ...");

			# Open file to store records
			open (DBRECFILE, ">>".$LogDBRecFile);

			#Record title line for easy importing into some spreadsheet programs
			printf DBRECFILE "Record,Date,Reading,Temperature,Battery\n";

			#Point to first record
			$LogCurrentRecord=0;

			# Get all records
			while (($CancelLogRetrieval==0) && ($LogCurrentRecord<$LogDBNextPtr)){
				my $Text = sendget({sqmcommand=>sprintf("L4%010dx\r",$LogCurrentRecord++)});
				if ($Text ne "") {
					@array=split(/,/,$Text);
					printf DBRECFILE sprintf(
						"%d,%s,%1.2f,%1.1f,%1.2f\n",
						$LogCurrentRecord,
						$array[1],
						$array[2],
						$array[3],
						(2.048 + (3.3 * $array[4])/256.0)
					);
				}
				#Allow the cancel button to be pressed
				$mw->update;
			}

			close (DBRECFILE);

			#Inform the user done, and where the file is stored.
			# my $dir=getcwd();
			my $Filepath=abs_path($LogDBRecFile);
			my $response = $mw->messageBox(
				-message=>"Logged database records have been stored in $Filepath",
				-type=>'ok',
				-icon=>'info'
			);
			$LogRecordResult->Contents("Database records saved to:\n$Filepath");
			LogRecord("Database records saved to: $Filepath");
		}
)->pack(-side => 'left');
	$b->attach($LogRetrieveAllButton,-balloonmsg=>"Copy all records from database over to disk file.");

# Cancel retrieving all records button
$LogCancelRetrieveButton=$LogStorageControlFrame->Button(
	-text=> "Cancel Retrieve",
	-command => sub {
			$CancelLogRetrieval=1;
		}
)->pack(-side => 'left');
	$b->attach($LogCancelRetrieveButton,-balloonmsg=>"Cancel retrieving records.");

# Log one record button
$LogDBOneButton=$LogStorageControlFrame->Button(
	-text=> "Log one",
	-command => sub {
			sendget({sqmcommand=>"L3x\r"});
			LogDBSize_get();
			$LogCurrentRecord=$LogDBNextPtr-1;
			LogRecordGet($LogCurrentRecord);
		}
)->pack(-side => 'left');
	$b->attach($LogDBOneButton,-balloonmsg=>"Log one record to database.");

# Erase button
$LogDBEraseButton=$LogStorageControlFrame->Button(
	-text=> "Erase all",
	-command =>sub {
			my $response = $mw->messageBox(
				-message=>"Erase entire database of records?\nAre you sure?\nCancel if not sure.",
				-type=>'okcancel',
				-icon=>'warning'
			);
			if ($response eq "Ok") {
				$LogRecordResult->Contents("Database being erased,\nplease wait a few seconds ...");
				$mw->update();
				sendget({sqmcommand=>sprintf("L2x\r")});
				$mw->update();
				sleep(2);
				LogRefreshButton_Click();
				$LogDBNextPtr=0;#should be done in refresh code
				$LogRecordResult->Contents("Database was just erased.");
				LogRecord("Database erased.");
			}
		}
)->pack(-side => 'left');
	$b->attach($LogDBEraseButton,-balloonmsg=>"Erase entire database.");

$LogRecordResult = $LoggingStorageFrame-> ROText (
	-relief => 'sunken',
	-height =>5,
	-width => 34,
	-font=> $code_font,
)->pack;

$LogNavigationFrame=$LoggingStorageFrame->Frame()->pack(-fill=>'x');
$LogNavigationTable=$LogNavigationFrame->Table(
	-rows=>1,
	-columns=>6,
	-scrollbars=>0,
)->pack(-anchor=>'w');

# get First record button
$LogFirstButton=$LogNavigationFrame->Button(
	-text=> "|<",
	-font=> $code_font,
	-command =>sub {
		$LogCurrentRecord=0;
		LogRecordGet($LogCurrentRecord);
	});
	$b->attach($LogFirstButton,-balloonmsg=>"Get first record of database.");
	$LogNavigationTable->put(0,0,$LogFirstButton);

# get Previous record button
$LogPreviousButton=$LogNavigationFrame->Button(
	-text=> "<",
	-font=> $code_font,
	-command =>sub {
		if (--$LogCurrentRecord<0) {$LogCurrentRecord=0;}
		LogRecordGet($LogCurrentRecord);
	});
	$b->attach($LogPreviousButton,-balloonmsg=>"Get previous record of database.");
	$LogNavigationTable->put(0,1,$LogPreviousButton);

# get Next record button
$LogNextButton=$LogNavigationFrame->Button(
	-text=> ">",
	-font=> $code_font,
	-command =>sub {
		if (++$LogCurrentRecord>$LogDBNextPtr-1) {$LogCurrentRecord=$LogDBNextPtr-1;} 
		LogRecordGet($LogCurrentRecord);
	});
	$b->attach($LogNextButton,-balloonmsg=>"Get next record of database.");
	$LogNavigationTable->put(0,2,$LogNextButton);

# get Last record button
$LogLastButton=$LogNavigationFrame->Button(
	-text=> ">|",
	-font=> $code_font,
	-command =>sub {
		$LogCurrentRecord=$LogDBNextPtr-1;
		LogRecordGet($LogCurrentRecord);
	});
	$b->attach($LogLastButton,-balloonmsg=>"Get last record of database.");
	$LogNavigationTable->put(0,3,$LogLastButton);

$LogDBSizeBarFrame = $LogNavigationTable->Frame();
$LogDBSizeBar = $LogDBSizeBarFrame->ProgressBar(
	-width => 18,
	-from => 0,
	-to => 100,
	-gap => 0,
	-variable => \$LogDBSizePB,
)->pack;
$LogNavigationTable->put(0,4,$LogDBSizeBarFrame);

$LogDBSizeLabel = $LogNavigationTable->Label(-text=> '%')->pack;
$LogNavigationTable->put(0,5,$LogDBSizeLabel);
	
sub LogRecordGet {
	$LogRecordResult->Contents("");
	my $Text = sendget({sqmcommand=>sprintf("L4%010dx\r",shift)});
		if ($Text ne "") {
		@array=split(/,/,$Text);
		$LogRecordResult->Contents(sprintf(
			"      Record: %d$EOL        Date: %s$EOL     Reading: %1.2fmpsas$EOL Temperature: %1.1fC$EOL     Battery: %1.2fV",
			$LogCurrentRecord+1,
			$array[1],
			$array[2],
			$array[3],
			(2.048 + (3.3 * $array[4])/256.0)
		));
	}
}
	
sub LogDBSize_get {
	$LogDBNextPtr = substr(sendget({sqmcommand=>"L1x\r"}),-6);
	my $LogDBSizePB=100.0*($LogDBNextPtr/$LogDBMax);
	$LogDBSizeLabel->configure(-text=>sprintf("%1.4f%%",$LogDBSizePB));
	$b->attach($LogDBSizeBarFrame,-balloonmsg=>sprintf("%d records of %d possible",$LogDBNextPtr,$LogDBMax));
	if ($LogDBNextPtr==0){
		$LogFirstButton->configure(-state=>'disabled');
		$LogNextButton->configure(-state=>'disabled');
		$LogPreviousButton->configure(-state=>'disabled');
		$LogLastButton->configure(-state=>'disabled');
	} else {
		$LogFirstButton->configure(-state=>'normal');
		$LogNextButton->configure(-state=>'normal');
		$LogPreviousButton->configure(-state=>'normal');
		$LogLastButton->configure(-state=>'normal');
	}
}
	
# --- Battery life -----------------------------------------------------------
$LoggingBatteryLifeFrame=$DataLoggingFrame2->LabFrame(
	-label=>"Estimated battery life"
)->pack(-fill=>'x');
$LoggingBatteryLifeTable=$LoggingBatteryLifeFrame->Table(
	-rows=>4,
	-columns=>2,
	-scrollbars=>0,
)->pack(-anchor=>'w');
$LoggingBatteryLifeTable->put(0,0,$LoggingBatteryLifeTable->Label(-text=>'Capacity',-anchor=>'e'));
$LoggingBatteryCapacityFrame=$LoggingBatteryLifeTable->Frame();
	$LoggingBatteryLifeTable->put(0,1,$LoggingBatteryCapacityFrame);
$LoggingBatteryCapacityEntry=$LoggingBatteryCapacityFrame->Entry(
	-relief => 'sunken',
	-width => 6,
	-font=> $code_font
	)->pack(-side=>'left');
	$b->attach($LoggingBatteryCapacityEntry,-balloonmsg=>"Enter your external battery capacity in mAH.");
	$LoggingBatteryCapacityFrame->Label(-text=>'mAH')->pack(-side=>'left');
$LoggingBatteryLifeTable->put(1,0,$LoggingBatteryLifeTable->Label(-text=>'Duration',-anchor=>'e'));
$LoggingBatteryLifeFrame=$LoggingBatteryLifeTable->Frame();
	$LoggingBatteryLifeTable->put(1,1,$LoggingBatteryLifeFrame);
$LoggingBatteryDuration = $LoggingBatteryLifeFrame -> ROText (
		-relief => 'sunken',
		-height => 1,
		-width  => 6,
		-font=> $code_font,
	)->pack(-side=>'left');
	$b->attach($LoggingBatteryDuration,-balloonmsg=>"Estimated battery lifetime based on logging settings.");
	$LoggingBatteryLifeUnitsText=$LoggingBatteryLifeFrame->Label(-text=>'time')->pack(-side=>'left');

$LoggingBatteryLifeTable->put(2,0,$LoggingBatteryLifeTable->Label(-text=>'Until',-anchor=>'e'));
$LoggingBatteryExpiryFrame=$LoggingBatteryLifeTable->Frame();
	$LoggingBatteryLifeTable->put(2,1,$LoggingBatteryExpiryFrame);
$LoggingBatteryExpiryDate = $LoggingBatteryExpiryFrame -> ROText (
		-relief => 'sunken',
		-height => 1,
		-width  => 19,
		-font=> $code_font,
	)->pack;
	$b->attach($LoggingBatteryExpiryDate,-balloonmsg=>"Estimated battery expiry date.");

$LoggingBatteryLifeTable->put(3,0,$LoggingBatteryLifeTable->Label(-text=>'Voltage',-anchor=>'e'));
$LoggingBatteryLifeFrame=$LoggingBatteryLifeTable->Frame();
	$LoggingBatteryLifeTable->put(3,1,$LoggingBatteryLifeFrame);
$LoggingBatteryVoltage = $LoggingBatteryLifeFrame -> ROText (
		-relief => 'sunken',
		-height => 1,
		-width  => 5,
		-font=> $code_font,
	)->pack(-side=>'left');
	$b->attach($LoggingBatteryVoltage,-balloonmsg=>"Measured system voltage.");
	$LoggingBatteryLifeFrame->Label(-text=>'V')->pack(-side=>'left');
sub LogBatteryVoltage_get {
	my $ADCValue = substr(sendget({sqmcommand=>"L5x\r"}),-3);
	my $BatteryVoltage = (2.048 + (3.3 * $ADCValue)/256.0);
	# printf ("Battery = %1.2fV\n",$BatteryVoltage);
	$LoggingBatteryVoltage->Contents(sprintf("%1.2f\n",$BatteryVoltage));
}



# ----------------------------------------------------------------------------
# For debugging purposes to put unit to sleep from PC connection.
# The unit automatically goes to sleep when running from battery only.
#
# $LRTCSLEEP=$Page4->Button(-text=> "Sleep",-command =>\&RTCSleepButton_Click);
# $LRTCSLEEP->pack(-in=>$Page4,-anchor=>'w');

# === GPS page ==========================================================
$GPSPageTable=$Page6->Table(
	-rows=>1,
	-columns=>2,
	-scrollbars=>0,
);
$GPSPageTable->pack(-in=>$Page6,-anchor=>'w');

$GPSButtonsTable=$GPSPageTable->Table(
	-rows=>2,
	-columns=>1,
	-scrollbars=>0,
);
$GPSPageTable->put(0,0,$GPSButtonsTable);

$GPSReqButton=$Page6->Button(-width=>10,-text=> "Time Loc.",-command =>\&GPSRequestButton_Click);
  $b->attach($GPSReqButton,-balloonmsg=>"Time and location from the GPS module");
  $GPSButtonsTable->put(0,0,$GPSReqButton);

$GPSGSVReqButton=$Page6->Button(-width=>10,-text=> "GSV",-command =>\&GPSGSVRequestButton_Click);
  $b->attach($GPSGSVReqButton,-balloonmsg=>"Satellites in View");
  $GPSButtonsTable->put(1,0,$GPSGSVReqButton);

$GPSResult = $Page6-> ROText (
	-relief => 'sunken',
	-height =>18,
	-width => 110,
	-font=> $code_font,
);
  $GPSPageTable->put(0,1,$GPSResult);

# === Configure page ==========================================================
if($ConfigOption == 1) {
	$LCALB=$Page5->Button(-text=> "Light Calibration",-command =>\&LightCalibrationButton_Click);
	$DCALB=$Page5->Button(-text=> "Dark Calibration",-command =>\&DarkCalibrationButton_Click);
	$LCALB->pack(-in=>$Page5,-anchor=>'w');
	$DCALB->pack(-in=>$Page5,-anchor=>'w');
}

# === Debug page ==========================================================
if($DebugPageOption== 1) {
# Future optional page
}


# --- Pack geometry ------------------------------------------------------------
$DataNoteBookFrame->pack(-expand=>1,-fill=>'both',-padx=>4,-pady=>4);
$DataNoteBook->pack(-in=>$DataNoteBookFrame,-fill=>'both',-expand=>1);

$Page0Frame->pack(-in=>$Page0,-fill=>'x');
$VersionFrame->pack(-in=>$Page0Frame,-side=>'left',-fill=>'x');
$VersionButton->pack(-in=>$VersionFrame,-fill=>'x');
$VersionResult->pack(-in=>$VersionFrame);

$ReadingFrame->pack(-in=>$Page0Frame,-side=>'left',-anchor=>'w');
$ReadingButton->pack(-in=>$ReadingFrame,-fill=>'x');
$ReadingResult->pack(-in=>$ReadingFrame);

$IIB->pack(-in=>$Page2,-anchor=>'w');
$IntervalTable->pack(-in=>$Page2,-anchor=>'w');

$FirmwareTable->pack(-in=>$Page3,-anchor=>'w');

$StatusBar ->pack (-side=>'bottom',-fill=>'x');
$ProgressBar-> pack(-in=>$StatusBar,-side=>'right');
$StatusText-> pack(-in=>$StatusBar,-side=>'left',-fill=>'x',-expand=>1);

# --- Start program --------------------------------------------------
FindDevicesButton_Click(); # Check for attached deives right away

MainLoop;


# ----- GUI I/O subroutines ---------

#Upon changing selected device, all results should be cleared
sub ClearResults {
	$ReadingResult->Contents("");
	$VersionResult->Contents("");
	$SCLOR->Contents("");
	$SLCTR->Contents("");
	$SDCPR->Contents("");
	$SDCTR->Contents("");
	$ITiE->Contents("");
	$ITiR->Contents("");
	$IThE->Contents("");
	$IThR->Contents("");
	$StatusText->Contents("");
	$PortEntry->configure(-background=>$PortEntryBackground);
	$EthIPEntry->configure(-background=>$EthIPEntryBackground);
	$EthPortEntry->configure(-background=>$EthPortEntryBackground);
	$CheckLockResult->Contents("");
	$SelectedModel=0;
	$LRTCTime->Contents("");
	$LogMode=-1;
	$LIThresholdE->delete(0,'end');
	$LISecondsE->delete(0,'end');
	$LIMinutesE->delete(0,'end');
}
 
sub StatusMessage {
	my $StatusMessageText = shift;
	$StatusText->Contents($StatusMessageText);
	if($ExtendedLogging == 1) {
		LogRecord($StatusMessageText);
	}
	$mw->update;
}
sub DevicesFound_Select {
	if ($DevicesFound->size()>=1) {
		SelectDevice($DevicesFound->curselection());
	}
}

sub SelectDevice {
	ClearResults();
	my $SelectionPointer = shift;
	if ($FoundUnits[$SelectionPointer][2] eq "USB") {
		$SelectingDeviceFlag=1;
		$CommNoteBook->raise('USB');
		$SelectingDeviceFlag=0;
		$PortEntry->delete(0,'end');
		$PortEntry->insert(0,$FoundUnits[$SelectionPointer][1]);
		$SerialNumberText->Contents($FoundUnits[$SelectionPointer][0]);

	} elsif ($FoundUnits[$SelectionPointer][2] eq "ETH") {
		$SelectingDeviceFlag=1;
		$CommNoteBook->raise('Ethernet');
		$SelectingDeviceFlag=0;
		$EthIPEntry->delete(0,'end');$EthIPEntry->insert(0,$FoundUnits[$SelectionPointer][1]);
		$EthIPEntry->configure(-background=>$EthIPEntryBackground);
		$MACText->Contents($FoundUnits[$SelectionPointer][0]);
		$EthPortEntry->delete(0,'end');$EthPortEntry->insert(0,"10001");
	}
}

sub FindDevicesButton_Click {
	# Notify that a search for devices is taking place.
	$DevicesFound->delete(0,"end");
	$DevicesFound->insert("end","Searching for devices, please wait ...");
	$mw->update;

	# Find and display USB attached devices.
	findusb();
	StatusMessage(sprintf("Finished searching for USB attached devices."));

	# Find and display Ethernet attached devices.
	findeth();
	StatusMessage(sprintf("Finished searching for Ethernet attached devices."));

	#Clear list in preparation for new device search.
	$DevicesFound->delete(0,"end");

	# Display; device type (USB/ETH), Identifier (serial_number/MAC), access point (COM_port/IP).
	foreach (@FoundUnits) {
		$DevicesFound->insert("end",sprintf("%s : %12s : %s",@$_[2],@$_[0],@$_[1]));
	}

	# If only one device exists, select it.
	if ($DevicesFound->index("end")==1) {
		SelectDevice(0);
	}
}

sub PortEntry_Modify {
	#Check if entered text matches a found device, blank serial number otherwise.
	$SerialNumberText->Contents("");
	foreach (@FoundUnits){
		if (@$_[1] eq $PortEntry->get()) {
			$SerialNumberText->Contents(@$_[0]);
		}
	}
}

sub EthIPText_Change {
	$MACText->Contents("");
	ClearResults();
}

sub CommPageSel {
	#Manually selecting between communication types clears results and selection pointer,
	# suppress action when selecting of device causes communication device type changing.
	if ($SelectingDeviceFlag ne 1) {
		ClearResults();
		$DevicesFound->selectionClear(0, 'end')
	}
}

sub ReadingButton_Click {
	if ($SelectedModel == 0) { #no model selected
		VersionButton_Click();
	}
	$ReadingResult->Contents("");
	my $Text = sendget({sqmcommand=>"rx\r"});
#  	$Text=substr($Text,0,length($Text)-2);#remove CR LF (for Windows only)
	if ($Text ne "") {
		@array=split(/,/,$Text);
		if ($SelectedModel == 0) { #no model selected
			$ReadingResult->Contents(sprintf("Could not get version."));
		}
		#3=Standard SQM-LE/LU, 5=SQM-LR, 6=SQM-LU-DL, 7 =SQM-LU-GPS 
		elsif (($SelectedModel==3) or ($SelectedModel==5) or ($SelectedModel==6)or ($SelectedModel==7)) {
			$ReadingResult->Contents(sprintf(
				"  Reading: %1.2fmpsas\nFrequency: %dHz$EOL  Counter: %dcounts$EOL     Time: %1.3fs$EOL     Tint: %1.1fC",
				$array[1],
				$array[2],
				$array[3],
				$array[4],
				$array[5],
			));
		}
		#8=Magnetometer
		elsif ($SelectedModel == 8) {
			$ReadingResult->Contents(sprintf(
				"M1: %dc\nT1: %10.7fC",
				$array[1],
				$array[2]/128.0,
			));
		}
		#1=ADA
		elsif ($SelectedModel == 1) {
			$ReadingResult->Contents(sprintf(
				"Frequency1: %dHz$EOL  Counter1: %dcounts$EOL     Time1: %1.3fs$EOL  Counter2: %dcounts$EOL     Time2: %1.3fs",
				$array[1],
				$array[2],
				$array[3],
				$array[4],
				$array[5],
			));
		}
		#4=Colour
		elsif ($SelectedModel == 4) {
		}
	}
}

sub VersionButton_Click {
	$VersionResult->Contents("");
	my $Text = sendget({sqmcommand=>"ix\r"});
# 	$Text=substr($Text,0,length($Text)-2);#remove CR LF  (for Windows only)
	if ($Text ne "") {
		@array=split(/,/,$Text);
		$VersionResult->Contents(sprintf(
			"Protocol: %d$EOL   Model: %d$EOL Feature: %d$EOL  Serial: %d",
			$array[1],
			$array[2],
			$array[3],
			$array[4]
		));
		$SelectedModel=$array[2];
		#Check selected model for optional tabs to be displayed

		#Datalogging tab: 6=SQM-LU-DL, 7 =SQM-LU-GPS 
		if (($SelectedModel==6)or ($SelectedModel==7)) {
			$DataNoteBook->pageconfigure( "page4", -state=>'normal' );
			$DataLoggingAvailable=1;

		} else {
			$DataNoteBook->pageconfigure( "page4", -state=>'disabled' );
			$DataLoggingAvailable=0;
		}
		#GPS tab: 7 =SQM-LU-GPS 
		if ($SelectedModel==7) {
			$DataNoteBook->pageconfigure( "page6", -state=>'normal' );

		} else {
			$DataNoteBook->pageconfigure( "page6", -state=>'disabled' );
		}
		
	}
}

sub GPSRequestButton_Click {
	my $Text = sendget({sqmcommand=>"g0x\r"});
	my @array0 = split(/[,*]/,$Text);
	my $Text = sendget({sqmcommand=>"g4x\r"});
	my @array1 = split(/[,*]/,$Text);

	if ($array2[6]=="0") {
		$Message = "Not available or invalid";
	} else {
		$Message = "Available";
	}

	$GPSResult->Contents(sprintf(
		"        Date: %10s       ddmmyy\n".
		"    UTC Time: %10s   hhmmss.sss\n".
		"    Latitude: %10s %1s  ddmm.mmmm\n".
		"   Longitude: %10s %1s dddmm.mmmm\n".
		"MSL Altitude: %10s%s\n".
		"         Fix: %s\n",
		$array1[9],
		$array0[1],
		$array0[2],$array0[3],
		$array0[4],$array0[5],
		$array0[9],lc($array0[10]),
		$Message
	));
}

sub GPSGSVRequestButton_Click {
	my $saw1 = sendget({sqmcommand=>"gAx\r"});
	my $saw2 = sendget({sqmcommand=>"gBx\r"});
	my $saw3 = sendget({sqmcommand=>"gCx\r"});

	@array1 = split(/[,*]/,$saw1);
	@array2 = split(/[,*]/,$saw2);
	@array3 = split(/[,*]/,$saw3);

	if ( $array1[1] == "1"){ @array2 = (); @array3 = ();}
	if ( $array1[1] == "2"){ @array3 = (); }

	$GPSResult->Contents(sprintf(
		"=========================== GSV  GNSS Satellites in View ===========================\n".
		"             Message ID: %7s %7s %7s GGA protocol header\n".
		"Total number ofmessages: %7s %7s %7s Range 1 to 3\n".
		"         Message number: %7s %7s %7s Range 1 to 3\n".
		"     Satellites in view: %7s %7s %7s \n\n".
		"           Satellite ID: %7s %7s %7s Channel 1 (Range 01 to 32)\n".
		"              Elevation: %7s %7s %7s degrees Channel 1 (Range 00 to 90)\n".
		"                Azimuth: %7s %7s %7s degrees Channel 1 (Range 000 to 359)\n".
		"             SNR (C/No): %7s %7s %7s dB-Hz Channel 1 (Range 00 to 99, null when not tracking)\n\n".
		"           Satellite ID: %7s %7s %7s Channel 4 (Range 01 to 32)\n".
		"              Elevation: %7s %7s %7s degrees Channel 4 (Range 00 to 90)\n".
		"                Azimuth: %7s %7s %7s degrees Channel 4 (Range 000 to 359)\n".
		"             SNR (C/No): %7s %7s %7s dB-Hz Channel 4 (Range 00 to 99, null when not tracking)\n",
		,$array1[0],$array2[0],$array3[0]
		,$array1[1],$array2[1],$array3[1]
		,$array1[2],$array2[2],$array3[2]
		,$array1[3],$array2[3],$array3[3]
		,$array1[4],$array2[4],$array3[4]
		,$array1[5],$array2[5],$array3[5]
		,$array1[6],$array2[6],$array3[6]
		,$array1[7],$array2[7],$array3[7]
		,$array1[8],$array2[8],$array3[8]
		,$array1[9],$array2[9],$array3[9]
		,$array1[10],$array2[10],$array3[10]
		,$array1[11],$array2[11],$array3[11]
	));

}

sub CalibrationInfoButton_Click {
	$SCLOR->Contents("");
	$SLCTR->Contents("");
	$SDCPR->Contents("");
	$SDCTR->Contents("");
	my $Text = sendget({sqmcommand=>"cx\r"});
# 	$Text=substr($Text,0,length($Text)-2);#remove CR LF 
	if ($Text ne "") {
		@array=split(/,/,$Text);
		$SCLOR->Contents(sprintf("%1.2fm",$array[1]));
		$SLCTR->Contents(sprintf("%1.1fC",$array[3]));
		$SDCPR->Contents(sprintf("%1.3fs",$array[2]));
		$SDCTR->Contents(sprintf("%1.1fC",$array[5]));
	}
}

sub CalibrationLogButton_Click {
	my $InformationText = sendget({sqmcommand=>"ix\r"});
	my $CalibrationText = sendget({sqmcommand=>"cx\r"});
	open (CALFILE, '>>calibrated.txt');
	($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=localtime(time);
	printf CALFILE "------- %4d-%02d-%02d %02d:%02d:%02d -------\n",$year+1900,$mon+1,$mday,$hour,$min,$sec;
	if ($CommNoteBook->raised() eq "USB") {
		print CALFILE $SerialNumberText->Contents();
	}
	print CALFILE $InformationText."\n";
	print CALFILE $CalibrationText."\n";
	close (CALFILE);
}

sub LightCalibrationButton_Click {
	my $Text = sendget({sqmcommand=>"zcalAx\r"});
}

sub DarkCalibrationButton_Click {
	my $Text = sendget({sqmcommand=>"zcalBx\r"});
}

sub SetLightCalOffsetButton_Click {
	$SCLOR->Contents("");
	my $Text = sendget({sqmcommand=>sprintf("zcal5%011.2fx\r",$SCLOTF->get())});
	if ($Text ne "") {
		@array=split(/,/,$Text);
	# 	@array=split(/,/,substr($Text,0,length($Text)-2));#remove CR LF and split
		$SCLOR->Contents(sprintf("%1.2fm",$array[2]));#Fill field
	}
}
sub SetLightCalTempButton_Click {
	$SLCTR->Contents("");
	my $Text = sendget({sqmcommand=>sprintf("zcal6%011.2fx\r",$SLCTTF->get())});
	if ($Text ne "") {
		@array=split(/,/,$Text);
	# 	@array=split(/,/,substr($Text,0,length($Text)-2));#remove CR LF and split
		$SLCTR->Contents(sprintf("%1.1fC",$array[2]));#Fill field
	}
}
sub SetDarkCalPerButton_Click {
	$SDCPR->Contents("");
	my $Text = sendget({sqmcommand=>sprintf("zcal7%011.3fx\r",$SDCPTF->get())});
	if ($Text ne "") {
		@array=split(/,/,$Text);
	# 	@array=split(/,/,substr($Text,0,length($Text)-2));#remove CR LF and split
		$SDCPR->Contents(sprintf("%2.3fs",$array[2]));#Fill field
	}
}
sub SetDarkCalTempButton_Click {
	$SDCTR->Contents("");
	my $Text = sendget({sqmcommand=>sprintf("zcal8%011.2fx\r",$SDCTTF->get())});
	if ($Text ne "") {
	# 	@array=split(/,/,substr($Text,0,length($Text)-2));#remove CR LF and split
		@array=split(/,/,$Text);
		$SDCTR->Contents(sprintf("%2.1fC",$array[2]));#Fill field
	}
}

sub IntervalInfo {
	$ITiE->Contents("");
	$ITiR->Contents("");
	$IThE->Contents("");
	$IThR->Contents("");
	my $Text = shift;
	if ($Text ne "") {
	# 	$Text=substr($Text,0,length($Text)-2);#remove CR LF 
		@array=split(/,/,$Text);
		$ITiE->Contents(sprintf("%ds",$array[0]));
		$ITiR->Contents(sprintf("%ds",$array[1]));
		$IThE->Contents(sprintf("%1.2fm",$array[2]));
		$IThR->Contents(sprintf("%1.2fm",$array[3]));
	}
}
sub IntervalInfoButton_Click {
	IntervalInfo(sendget({sqmcommand=>"Ix\r"}));
}
sub IntervalTimeERButton_Click {
	IntervalInfo(sendget({sqmcommand=>sprintf("P%010dx\r",$ITiTF->get())}));
}
sub IntervalTimeRButton_Click {
	IntervalInfo(sendget({sqmcommand=>sprintf("p%010dx\r",$ITiTF->get())}));
}
sub ThresholdERButton_Click {
	IntervalInfo(sendget({sqmcommand=>sprintf("T%011.2fx\r",$IThTF->get())}));
}
sub ThresholdRButton_Click {
	IntervalInfo(sendget({sqmcommand=>sprintf("t%011.2fx\r",$IThTF->get())}));
}

sub RTCSetButton_Click {
	my $OrigTime = substr(sendget({sqmcommand=>"Lcx\r"}),-19);
	# prepare current date time to be sent to unit
	($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=localtime(time);
	my $NewTime = substr(sendget({sqmcommand=>sprintf("LC%2s-%02d-%02d %1d %02d:%02d:%02dx\r",
		substr($year+1900,2,2),
		$mon+1,
		$mday,
		$wday+1,
		$hour,
		$min,
		$sec)}),-19);
	$LRTCTime->Contents($NewTime);
	LogRecord("Setting RTC. Was: $OrigTime, Now: $NewTime");
}

sub LogRefresh {
	$LISecondsE->delete(0,'end');
	$LIMinutesE->delete(0,'end');
	$LIThresholdE->delete(0,'end');
	# $LIFWE->Contents("");
	my $Text = shift;
	if ($Text ne "") {
		@array=split(/,/,$Text);
		$LISecondsE->insert(0,sprintf("%d",$array[1]));
		$LIMinutesE->insert(0,sprintf("%d",$array[2]));
		$LISecLabel->configure(-text=>sprintf("seconds (always on) (countdown=%ds)",$array[3]));
		$LIMinLabel->configure(-text=>sprintf("minutes (power save mode) (countdown=%dm)",$array[4]));
		$LIThresholdE->insert(0,sprintf("%1.2f",$array[5]));
		}
}

sub LogSettingsApplyButton_Click {
	sendget({sqmcommand=>sprintf("LPS%010dx\r",$LISecondsE->get())});
	sendget({sqmcommand=>sprintf("LPM%010dx\r",$LIMinutesE->get())});
	sendget({sqmcommand=>sprintf("LT%011.2fx\r",$LIThresholdE->get())});
	$LogMode=substr(sendget({sqmcommand=>sprintf("LM%dx\r",$LogMode)}),3,1);
}
sub LogRefreshButton_Click {
	my $OrigTime = substr(sendget({sqmcommand=>"Lcx\r"}),-19);
	$mw->update;
	$LRTCTime->Contents($OrigTime);
	LogRecord("Refresh Logging info. SQM RTC was: $OrigTime");
	LogRefresh(sendget({sqmcommand=>"LIx\r"}));
	$LogMode=substr(sendget({sqmcommand=>sprintf("Lmx\r")}),3,1);
	print $LogMode;
	LogDBSize_get;
	LogBatteryVoltage_get;
	# sendget({sqmcommand=>sprintf("Lax\r")});#get alarm settings
}

# sub RTCSleepButton_Click {
	# sendget({sqmcommand=>"Lsx\r"});
# }

sub SelectFirmwareButton_Click {
	$SFR->configure(-background=>$SFRBackground);
	my $FIRMWARE_FILE;
	my $types = [
	['Update files',     '.hex'],
	['All Files',        '*',  ],
	];

	if ($^O eq "MSWin32"){
		my $dir = getcwd();
		$dir =~ s/\//\\/; #Windows needs backslashes
		$FIRMWARE_FILE  = $Page3->getOpenFile(
			-title => "Select Firmware Upgrade File", 
			-filetypes=>$types,
			-initialdir=>$dir
			);
			# Directory handling for Windows OS
			# 	use Cwd;
			# 	my $dir = getcwd();
			# 	$dir =~ s/\//\\/; #Windows needs backslashes
			# 	#print $dir."\n";
	} else {
		$FIRMWARE_FILE  = $Page3->getOpenFile(
			-title => "Select Firmware Upgrade File", 
			-filetypes=>$types
		);
	}

	if (defined $FIRMWARE_FILE) {
		$SFR->Contents($FIRMWARE_FILE);
	}
}

sub LoadFirmwareButton_Click {

	OpenComm();
	
	my $FIRMWARE_FILE = $SFR->get("1.0",'end-1chars');
	my $FileObj = open(DAT, $FIRMWARE_FILE);
	if (defined $FileObj) {
		$SFR->configure(-background=>$SFRBackground);
	} else {
		$SFR->configure(-background=>'red');
		StatusMessage("Error: Couldn't open firmware file $FIRMWARE_FILE");
		return "";;
	}

	StatusMessage("Resetting unit ...");
	$port->write(chr(0x19));
	sleep 1;
	StatusMessage("Unit should have been reset ...");

	StatusMessage("Opened firmware file for transferring ...");
	@raw_data=<DAT>;

 	my $output_string;
	my $arraysize = @raw_data;
	$ProgressBar->configure(-to=>$arraysize);
	my $line_number=0;
	StatusMessage("Transferring firmware to unit, DO NOT INTERRUPT  ...");
	foreach $_ (@raw_data) {
		$sqmcommand = $_;
		#strip off all trailing white spaces - tabs, spaces, new lines and returns
		$sqmcommand =~ s/\s+$//;
		# Add required newline character to intel.hex string
		$sqmcommand .= "\n";


		if (sendget({sqmcommand=>$sqmcommand,leaveopen=>1}) =~ "Ok") {
# 			print "Sent packet ",($line_number + 1),"/$arraysize\n";
			$Progress=$line_number++;
			$mw->update;
			#Win32::GUI::DoEvents();
		} else{
			print "Timed out waiting for response to: $output_string\n";
			StatusMessage("Timed out waiting for response to: $output_string.");
			$port->close;
			undef $port;
			return;
		}
	}
	StatusMessage("Transfer completed.");
	$Progress=0;

	$port->close;
	undef $port;
}

sub CheckLockButton_Click {
	$CheckLockResult->Contents("");
	my $Text = sendget({sqmcommand=>sprintf("zcalDx\r")});
	if ($Text ne "") {
		if (substr($Text,3,1) eq "L") {
			$CheckLockResult->Contents("Locked");
		} else {
			$CheckLockResult->Contents("Unlocked");
		}
	}
}

# Find USB attached FTDI devices
sub findusb {
	#Define empty array to hold all found SQM devices
	@FoundUnits=();

	if ($^O eq "MSWin32") {

		# use cpan to install Win32::SerialPort as follows:
		# Start menu, All Programs, Strawberry Perl, Tools, CPAN Client, install Win32::SerialPort, exit
		# If it fails, then search in C:\strawberry\... for serialport
		#   then manually copy SerialPort.pm into C:\strawberry\perl\site\lib\Win32 (create Win32 directory if necessary)
		#   and copy CommPort.pm into C:\strawberry\perl\site\lib\Win32API (create Win32API directory if necessary)

		# Find FTDI USB device on a Windows machine
		$Regkey="HKEY_LOCAL_MACHINE\\SYSTEM\\CurrentControlSet\\Enum\\";
		$RegCOMkey="HKEY_LOCAL_MACHINE\\HARDWARE\\DEVICEMAP\\SERIALCOMM\\";
	
		# Check if FTDI "key" exists.
		# This is an indication that either the device is not installed or that the FTDI driver is not installed.
		$FTDIExists=0;
		StatusMessage(sprintf("Checking Registry for: %s ...",$Regkey));
		$FTDIKey= $Registry->Open($Regkey,{Access=>KEY_READ});
		foreach $FTDIKeyentry (  keys(%$FTDIKey)  ) {
			if ($FTDIKeyentry =~ /FTDIBUS/){
				$FTDIExists++;
			}
		}
		StatusMessage(sprintf("Fount %d FTDI device(s).",$FTDIExists));
		if ($FTDIExists==0){
			print "\n\tThe FTDI driver or SQM-LU device is not installed.\n";
			print "\n\tMake sure that the device is attached and that the FTDI VCP\n";
			print "\tdriver has been installed.\n";
			print "\n\tThe latest driver can be obtained from ftdichip.com/Drivers/VCP.htm\n";
			StatusMessage(sprintf("FTDI driver for SQM USB not installed"));
			return;
		}
	
	
		#Check active COM ports for VCP devices
		StatusMessage(sprintf("Checking Registry for: %s ...",$RegCOMkey));
		$COMKey= $Registry->Open($RegCOMkey,{Access=>KEY_READ()});
		if (!defined $COMKey) {
			print "Can't read SERIALCOMM key: $^E\n";
			StatusMessage(sprintf("Can't read SERIALCOMM key: $^E"));
			return;
		}
		
		$found=0;
		foreach $entry (  keys(%$COMKey)  ) {
			if ($entry =~ /Device\\VCP/){
				#print "FTDI VCP device found at: ".$Registry->Open($RegCOMkey,{Access=>KEY_READ()})->{$entry}."\n";
				$found++;
			}
		}
		if ($found==0){
			StatusMessage(sprintf("No active VCP SQM USB devices found for."));
			return;
		}
	
		#Check defined FTDI COM port assignments for matches to Active COM ports
		$FTDIKeys= $Registry->Open($Regkey."FTDIBUS",{Access=>KEY_READ});
		foreach $entry (  keys(%$FTDIKeys)  ) {
		$FTDIKey= $Registry->Open($Regkey."FTDIBUS\\".$entry."\\0000\\",{Access=>KEY_READ});
			foreach $FTDIKeyentry (  keys(%$FTDIKey)  ) {
				if ($FTDIKeyentry =~ /Device Parameters/){
					$somekeys= $Registry->Open($Regkey."FTDIBUS\\".$entry."\\0000\\".$FTDIKeyentry,{Access=>KEY_READ});
					foreach $somekey (  keys(%$somekeys)  ) {
						if ($somekey =~ /PortName/){
						#Check through all active com ports
							foreach $COMentry (  keys(%$COMKey)  ) {
								#Filter VCP com ports
								if ($COMentry =~ /Device\\VCP/){
									$ActiveCOM= $Registry->Open($RegCOMkey,{Access=>KEY_READ()})->{$COMentry};
									$DefinedCOM= $somekeys->{$somekey};
									#Filter the matches of active com ports to the FTDI defined com ports
									if ($ActiveCOM eq $DefinedCOM) {
										$DefinedSerial=substr($entry,18,8);
										#print "".$DefinedSerial."\t".$DefinedCOM."\n";
										push @FoundUnits,[($DefinedSerial,$DefinedCOM,"USB")];
									}
								}
							}
						}
					}
				}
			}
		}
	} else {
	
		# Find FTDI USB device on a Unix machine
		
		open (LSHAL, "lshal |");
		$LookForState=0;
		while (<LSHAL>){
		
			#Look for serial number:
			#udi = '/org/freedesktop/Hal/devices/usb_device_403_6001_FTSC89JG_if0_serial_usb_0'
			if ($LookForState==0 and $_ =~ m/^udi =/ and $_ =~ m/_403_6001_/){
				#print "\tfound serial=".$_;
				$LookForState=1;
			}
			
			#Look for:
			#  usb_device.serial = 'FTSC89JG'  (string)
			if ($LookForState==1 and $_ =~ m/usb.serial/){
				@array = split(" ",$_);
				$USBDeviceSerial=$array[2];
				$USBDeviceSerial =~ s/^.(.*).$/$1/; #Strip first and last character
# 				print "Serial=".$USBDeviceSerial."\t";
				$LookForState=2;
			}
			
			#Look for linux.device_file:
			$LinuxDeviceFile="";
			if ($LookForState==2 and $_ =~ m/linux.device_file/){
				@array = split(" ",$_);
				$LinuxDeviceFile =$array[2];
				$LinuxDeviceFile =~ s/^.(.*).$/$1/; #Strip first and last character
# 				print "Dev=".$LinuxDeviceFile."\n";
				$LookForState=0;
			}
			if ($LinuxDeviceFile ne "") {
				push @FoundUnits,[($USBDeviceSerial,$LinuxDeviceFile,"USB")];
			}
		}
		close LSHAL;
	}
}

#Find Ethernet attached Lantronix Xport devices
sub findeth {
	# Make a socket for network communications.
	my $handle = IO::Socket::INET->new(
		Proto => 'udp',
	);
	
	# Prepare for network broadcast.
	setsockopt($handle, SOL_SOCKET, SO_BROADCAST, 1);
	
	# Broadcast address is everyone.
	my $HOSTNAME="255.255.255.255";
	
	# Lantronics XPORT special port for replying to unit search.
	my $PORTNO=30718;
	
	# Prepare IP and Port number for use later.
	my $ipaddr   = inet_aton($HOSTNAME);
	my $portaddr = sockaddr_in($PORTNO, $ipaddr);
	
	# Put data into the proper format for sending over the network.
	#  N = An unsigned long (32-bit) in "network" (big-endian) order.
	#  f6 = the request recognized by the Lantronix X-Port module.
	my $Request = pack("N", 0x000000f6);
	
	# Send the broadcasted search for units.
	send($handle, $Request, 0, $portaddr);# == length($Request);
	
	StatusMessage("Waiting for all Ethernet devices to respond ...");
	my $s = IO::Select->new($handle);

	while (length($s->can_read(2)) > 0) {
		# Receive one packet of 30 characters.
		my $portaddr = recv($handle, $ReceivedMessage, 30, 0);
		if (defined $portaddr) {
			# Gather information from received packet.
			($PORTNO, $ipaddr) = sockaddr_in($portaddr);

			# Pull MAC from message, convert HEX to ASCII and print it.
			my $MAC="";
			if (substr($ReceivedMessage,3,1) eq chr(0xf7)) {
				foreach (24 .. 29) {
					$MAC.=sprintf("%02X",ord(substr($ReceivedMessage,$_,1)));
				}
			}
			push @FoundUnits,[($MAC,inet_ntoa($ipaddr),"ETH")];
			StatusMessage(sprintf("Found: %s at %s",$MAC,inet_ntoa($ipaddr)));
		} else {
			print "Undefined portaddr.\n";
		}
	}
}

#Send string, get string back
sub sendget {
	my ($args) = @_;
	my $sqmcommand = $args->{sqmcommand};

	if (($CommNoteBook->raised() eq "USB") or ($CommNoteBook->raised() eq "RS232")) {
		#Do not open communications port if it was left open
		if (!defined $args->{leaveopen}) {
			if (!OpenComm()){return;}
		}
		$port->write($sqmcommand);
		
		# Initialize response string
		$saw="";

		while (1){
			($count,$character)=$port->read(1);
			if ($count eq 0){
				last; #Break out of while loop if no characters received
			} else {
				$saw .=$character;
				if ($character eq chr(0x0a)) {last;}
			}
		}

		#Do not close communications port if it is to be left open
		if (!defined $args->{leaveopen}) {
			$port->close;
			undef $port;
		}
	
	} elsif ($CommNoteBook->raised() eq "Ethernet") {

		# Initialize response string
		$saw="";
	
		#Do not open communications port if it was left open
		if (!defined $args->{leaveopen}) {
			if (!OpenComm()){return;}
		}
		# Send request to SQM
		$port->write($sqmcommand);
	
		my $s = IO::Select->new($port);
		#Try to read then Timeout for 2 seconds
		while ((length($s->can_read(2)) > 0)) {
			# Receive packet 1 character at a time.
			my $portaddr = recv($port, $ReceivedMessage, 1, 0);
			if (defined $portaddr) {
				$saw .= $ReceivedMessage;
			} else {
				print "Undefined portaddr.\n";
			}
			if ($saw =~ /\n/) {last;}
		}

		#Compose portname for reporting.
		$PortName = $EthIPEntry->get().":".$EthPortEntry->get();
		
		#Do not close communications port if it is to be left open
		if (!defined $args->{leaveopen}) {
			$port->close;
			undef $port;
		}
	} else {
		StatusMessage("Select a device or type first");
	}

	#strip off all trailing white spaces - tabs, spaces, new lines and returns
	$sqmcommand =~ s/\s+$//;
	$saw =~ s/\s+$//;

	StatusMessage(sprintf("Sent: %s   To: %s   Received: %s",$sqmcommand,$PortName,$saw));
	return $saw;
}

sub OpenComm {
	if ($CommNoteBook->raised() eq "USB") {
		$PortName=$PortEntry->get();	

		if ($^O eq "MSWin32"){
		 	$port = Win32::SerialPort->new($PortName);
		} else {
			$port = Device::SerialPort->new($PortName);
		}

		if (defined $port) {
			$PortEntry->configure(-background=>$PortEntryBackground);
		} else {
			$PortEntry->configure(-background=>'red');
			StatusMessage("Error: Couldn't open communications port $PortName");
			return "";;
		}

		$port->user_msg(ON);
		$port->baudrate(115200);
		$port->parity("none");
		$port->stopbits(1);
		$port->databits(8);
		$port->handshake("none");
		$port->write_settings || undef $port;
		$port->read_char_time(100); # Wait for each character x milliseconds
		$port->buffers(4096, 4096);
		#$port->handshake("rts"); #(maybe for RS232 only??)

	} elsif ($CommNoteBook->raised() eq "Ethernet") {

		# open socket
		StatusMessage("Opening Ethernet port ...");
		if (($EthIPEntry->get() ne "") && $EthPortEntry->get() ne "") {
			$port = IO::Socket::INET->new(
				PeerAddr => $EthIPEntry->get(),
				PeerPort => $EthPortEntry->get(),
				Proto    => 'tcp'
			);
			StatusMessage("Ethernet port opened ...");
		
			# Flush buffer after each print or write
			$port->autoflush(1);
		} else {
			StatusMessage("Ethernet port NOT opened!");
		}
	} elsif ($CommNoteBook->raised() eq "RS232") {

		$PortName=$RS232PortEntry->get();	
		my $BaudRate=$RS232BaudEntry->get();	

		if ($^O eq "MSWin32"){
		 	$port = Win32::SerialPort->new($PortName);
		} else {
			$port = Device::SerialPort->new($PortName);
		}

		if (defined $port) {
			$RS232PortEntry->configure(-background=>$RS232PortEntryBackground);
		} else {
			$RS232PortEntry->configure(-background=>'red');
			StatusMessage("Error: Couldn't open communications port $PortName");
			return "";;
		}
	
		$port->user_msg(ON);
		$port->baudrate($BaudRate);
		$port->parity("none");
		$port->stopbits(1);
		$port->databits(8);
		$port->handshake("none");
		$port->write_settings || undef $port;
		$port->read_char_time(100); # Wait for each character x milliseconds
		$port->buffers(4096, 4096);
		$port->handshake("rts");

	} else {
		StatusMessage("Select a device or type first");
	}
}

sub LogRecord {
	my $LogString = shift;
	open (LOGFILE, '>>logfile.txt');
	($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=localtime(time);
	printf LOGFILE "%4d-%02d-%02d %02d:%02d:%02d ",$year+1900,$mon+1,$mday,$hour,$min,$sec;
	print LOGFILE $LogString."\n";
	close (LOGFILE);
}
